# kb_acceptance_tests_v1.0.1.ps1
# Purpose: Acceptance check for BUG_KB_JSON pointer vs bootpack_paste (PS 5.1 safe)

# -- utils robustes (UTF-8 no BOM, CRLF, pas de syntaxes récentes) --
function Read-Text([string]$p){
  if(-not (Test-Path -LiteralPath $p)){ return '' }
  try { return Get-Content -LiteralPath $p -Raw -Encoding UTF8 } catch { return (Get-Content -LiteralPath $p -Raw) }
}
function To-IntSafe([object]$v){
  if($null -eq $v){ return -1 }
  if($v -is [array]){ if($v.Length -gt 0){ $v = $v[0] } else { return -1 } }
  try { return [int]$v } catch { return -1 }
}
function Count-EntriesFromJson([string]$jsonText){
  if([string]::IsNullOrWhiteSpace($jsonText)){ return -1 }
  if($jsonText.Length -gt 0 -and ([int]$jsonText[0] -eq 0xFEFF)){ $jsonText = $jsonText.Substring(1) } # strip BOM
  try{
    $obj = $jsonText | ConvertFrom-Json
    if($null -ne $obj -and $obj.PSObject.Properties.Name -contains 'entries'){ return @($obj.entries).Count }
  }catch{}
  try{
    $m = [regex]::Match($jsonText,'(?ms)"entries"\s*:\s*\[(.*?)\]')
    if($m.Success){ return ([regex]::Matches($m.Groups[1].Value,'(?i)"id"\s*:')).Count }
  }catch{}
  return -1
}
function Extract-PointerField([string]$raw,[string]$name){
  # robuste : ancre le bloc, tolère espaces, colonnes, NBSP; trim final
  $pat = '(?ms)^\[BUG_KB_JSON_POINTER\].*?^\s*' + [regex]::Escape($name) + '\s*:\s*(.+?)\s*$'
  $m = [regex]::Match($raw,$pat)
  if($m.Success){ return $m.Groups[1].Value.Trim() } else { return '' }
}

# -- lecture pointer depuis bootpack --
$root = $args[0]
$boot = Join-Path $root 'bootpack\bootpack.txt'
$raw  = Read-Text $boot

$P_path = Extract-PointerField $raw 'Path'
$P_sha  = Extract-PointerField $raw 'SHA256'
$P_entS = Extract-PointerField $raw 'Entries'
$P_szS  = Extract-PointerField $raw 'SizeBytes'

if($null -eq $P_path){ $P_path='' }
if($null -eq $P_sha ){ $P_sha ='' }
$P_sha  = ($P_sha -replace '\s','').ToUpperInvariant()
$P_ent  = To-IntSafe $P_entS
$P_sz   = To-IntSafe $P_szS

# -- mesures réelles sur le paste pointé --
if([string]::IsNullOrWhiteSpace($P_path) -or -not (Test-Path -LiteralPath $P_path)){
  Write-Host "[ERROR] KB Pointer path invalid or missing in bootpack."; exit 2
}
$ptrRaw = Read-Text $P_path
try{ $psSha = (Get-FileHash -Algorithm SHA256 -LiteralPath $P_path).Hash.ToUpperInvariant() }catch{ $psSha='' }
try{ $psSize = (Get-Item -LiteralPath $P_path).Length }catch{ $psSize = -1 }
$psEnt = Count-EntriesFromJson $ptrRaw

# -- comparaisons booléennes --
$eqPath = ($P_path -ceq (Join-Path $root 'bootpack\bootpack_paste.txt')) # chemin attendu “paste”
$eqSha  = ($P_sha -eq $psSha)
$eqEnt  = ($P_ent -eq $psEnt)
$eqSize = ($P_sz  -eq [int]$psSize)

Write-Host ("PTR   : path==paste={0} sha==paste={1} entries==paste={2} size==paste={3}" -f $eqPath,$eqSha,$eqEnt,$eqSize)
if($eqPath -and $eqSha -and $eqEnt -and $eqSize){
  Write-Host "Result: PASS"
}else{
  if(-not $eqPath){ Write-Host " - Pointer.Path mismatch" }
  if(-not $eqSha ){ Write-Host " - Pointer.SHA mismatch" }
  if(-not $eqEnt ){ Write-Host " - Pointer.Entries mismatch" }
  if(-not $eqSize){ Write-Host " - Pointer.SizeBytes mismatch" }
  Write-Host "Result: FAIL"
  exit 2
}

# DOC-VERSION-FOOTER: kb_acceptance_tests_v1.0.1.ps1 | Enc=UTF8-noBOM CRLF | _MAJ_=2025-10-22T18:00:00+02:00