# ============================================================
# Script : setup_gouvernance_v3a.ps1
# Objet  : Creation de la structure ChatGPT-Gouvernance-Projets
# Auteur : ChatGPT - GPT-5
# Date   : 11/10/2025
# Version: v3a (structure + verification automatique)
# ============================================================

# --- Parametres globaux ---
$BasePath = "\\DS-918\chatgpt\ChatGPT-Gouvernance-Projets"
$DocPath  = Join-Path $BasePath "00_DOCUMENTATION"
$RegPath  = Join-Path $BasePath "_registry"
$NowStamp = Get-Date -Format "dd-MM-yyyy HH:mm"
$HostName = $env:COMPUTERNAME

Write-Host "[INFO] Initialisation du projet de gouvernance..." -ForegroundColor Cyan

# --- Fonctions utilitaires ---
function Write-Utf8NoBom([string]$Path, [string]$Content) {
    $utf8NoBom = New-Object System.Text.UTF8Encoding($false)
    [System.IO.File]::WriteAllText($Path, $Content, $utf8NoBom)
}

function Normalize-Ascii([string]$text) {
    $formD = $text.Normalize([Text.NormalizationForm]::FormD)
    $sb = New-Object System.Text.StringBuilder
    foreach ($ch in $formD.ToCharArray()) {
        $cat = [Globalization.CharUnicodeInfo]::GetUnicodeCategory($ch)
        if ($cat -ne [Globalization.UnicodeCategory]::NonSpacingMark) {
            [void]$sb.Append($ch)
        }
    }
    $s = $sb.ToString()
    $s = $s -replace "[\u2018\u2019]","'"
    $s = $s -replace "[\u2013\u2014]","-"
    $s = $s -replace "\u2026","..."
    return $s.Normalize([Text.NormalizationForm]::FormC)
}

function Write-NormalizedFile([string]$Path, [string]$Content) {
    $norm = Normalize-Ascii $Content
    Write-Utf8NoBom -Path $Path -Content $norm
    Write-Host "[OK] Fichier cree : $Path" -ForegroundColor Yellow
}

# --- Creation des dossiers ---
$dirs = @(
    "$BasePath\00_DOCUMENTATION",
    "$BasePath\00_DOCUMENTATION\scripts_utilitaires",
    "$BasePath\01_SEEDBOX_MANAGER",
    "$BasePath\02_HUB_LYON_LUMIERE",
    "$BasePath\03_ARCHIVES_DISCUSSIONS",
    "$BasePath\03_ARCHIVES_DISCUSSIONS\SeedboxManager",
    "$BasePath\03_ARCHIVES_DISCUSSIONS\HubLyonLumiere",
    "$BasePath\03_ARCHIVES_DISCUSSIONS\Autres",
    "$BasePath\_registry"
)
$createdDirs = 0
foreach ($d in $dirs) {
    if (-not (Test-Path $d)) {
        New-Item -ItemType Directory -Path $d | Out-Null
        Write-Host "[OK] Dossier cree : $d" -ForegroundColor Green
        $createdDirs++
    }
}

# --- Contenu principal : fichiers texte ---
$docFiles = @{
"README_GOUVERNANCE.txt" = @"
CHATGPT - GOUVERNANCE DES PROJETS
---------------------------------
Ce projet definit la structure et les regles communes
pour tous les projets geres avec ChatGPT.

Objectifs :
 - Uniformiser les pratiques entre projets (Seedbox Manager, Hub Lyon Lumiere)
 - Garantir la tracabilite et la coherence documentaire
 - Servir de socle methodologique stable

Derniere mise a jour : $NowStamp
"@

"STRUCTURE_SYSTEME_GLOBAL.txt" = @"
STRUCTURE DU SYSTEME GLOBAL - VERSION INITIALE
===============================================
Objectif :
Definir l'organisation hierarchique generale des projets afin d'assurer une coherence durable entre :
- la gouvernance globale,
- les projets principaux (pro et perso),
- et leurs sous-projets respectifs.

Niveaux :
  0 : Gouvernance globale
  1 : Organisation globale / TechDoc globale
  2 : Projets principaux (PRO / PERSO)
  3 : Sous-projets concrets

Regles :
 - Les regles du niveau 0 s'appliquent a tous les niveaux inferieurs.
 - Chaque sous-projet reste autonome mais se refere a la gouvernance.
 - Les discussions lourdes sont archivees.
 - Les documents techniques heritent de la TechDoc globale.
"@

"ENCODAGE_ET_VERIFICATION_GUIDE.txt" = @"
GUIDE D'ENCODAGE ET DE VERIFICATION
===================================
1) CREATION
   Script : setup_gouvernance_v3a.ps1
   - Cree l'arborescence
   - Genere les fichiers en UTF-8 sans BOM

2) VERIFICATION
   Script : verify_encoding_gouvernance_fixed.ps1
   - Analyse encodage et presence de caracteres accentues
   - Rapport : _registry\ENCODING_REPORT.txt

3) CORRECTION
   Script : fix_encoding_gouvernance_fixed.ps1
   - Nettoie et re-ecrit les fichiers
   - Rapport : _registry\FIX_REPORT.txt

Bonnes pratiques :
 - Editer avec Notepad++ en UTF-8 sans BOM
 - Eviter Word et LibreOffice
 - Reexecuter verification apres ajout manuel de fichiers
"@
}

$createdFiles = 0
foreach ($file in $docFiles.Keys) {
    $path = Join-Path $DocPath $file
    Write-NormalizedFile -Path $path -Content $docFiles[$file]
    $createdFiles++
}

# --- Installation automatique du launcher run_encoding_maintenance.ps1 ---
$launcherScript = @"
# ============================================================
# Script : run_encoding_maintenance.ps1
# Objet  : Lance verification + correction + re-verification
# Auteur : ChatGPT - GPT-5
# Date   : 11/10/2025
# ============================================================

\$BasePath = "\\DS-918\chatgpt\ChatGPT-Gouvernance-Projets"
\$ToolsPath = Join-Path \$BasePath "00_DOCUMENTATION\scripts_utilitaires"
\$VerifyScript = Join-Path \$ToolsPath "verify_encoding_gouvernance_fixed.ps1"
\$FixScript    = Join-Path \$ToolsPath "fix_encoding_gouvernance_fixed.ps1"
\$ReportVerify = Join-Path \$BasePath "_registry\ENCODING_REPORT.txt"
\$NowStamp = Get-Date -Format "dd-MM-yyyy HH:mm"

Write-Host "============================================" -ForegroundColor Cyan
Write-Host "[RUN] Maintenance d'encodage - \$NowStamp" -ForegroundColor Cyan
Write-Host "============================================`n"

function Run-Script(\$scriptPath, \$label) {
    if (Test-Path \$scriptPath) {
        Write-Host "[INFO] Execution de \$label ..." -ForegroundColor Yellow
        & \$scriptPath
        Write-Host ""
    } else {
        Write-Host "[ERREUR] Script manquant : \$scriptPath" -ForegroundColor Red
        exit 1
    }
}

# Etape 1 : Verification initiale
Run-Script \$VerifyScript "Verification initiale"

# Lecture du rapport
\$needFix = \$false
if (Test-Path \$ReportVerify) {
    \$reportText = Get-Content -Path \$ReportVerify -Raw -Encoding UTF8
    if (\$reportText -match "AccentsDetectes=OUI" -or \$reportText -match "UTF-16" -or \$reportText -match "BOM") {
        \$needFix = \$true
        Write-Host "[ALERTE] Anomalies detectees dans les fichiers." -ForegroundColor Yellow
    } else {
        Write-Host "[CHECK] Aucun probleme d'encodage detecte." -ForegroundColor Green
    }
} else {
    Write-Host "[WARN] Rapport introuvable apres verification initiale." -ForegroundColor Yellow
}

# Etape 2 : Correction si necessaire
if (\$needFix) {
    Run-Script \$FixScript "Correction automatique"
} else {
    Write-Host "[INFO] Etape de correction sautee (aucune anomalie)." -ForegroundColor Cyan
}

# Etape 3 : Re-verification
Run-Script \$VerifyScript "Re-verification finale"

# Etape 4 : Affichage du resultat
if (Test-Path \$ReportVerify) {
    Write-Host "`n[OK] Rapport final : \$ReportVerify" -ForegroundColor Green
    Start-Process notepad.exe \$ReportVerify
} else {
    Write-Host "[WARN] Aucun rapport de verification final trouve." -ForegroundColor Yellow
}

Write-Host "`n[END] Maintenance complete." -ForegroundColor Cyan
"@

$launcherPath = "$DocPath\scripts_utilitaires\run_encoding_maintenance.ps1"
Write-Utf8NoBom -Path $launcherPath -Content $launcherScript
Write-Host "[OK] Launcher installe : $launcherPath" -ForegroundColor Yellow


# --- Registry et logs ---
if (-not (Test-Path $RegPath)) { New-Item -ItemType Directory -Path $RegPath | Out-Null }
# --- Registry et logs (mise à jour version 3b) ---
if (-not (Test-Path $RegPath)) { New-Item -ItemType Directory -Path $RegPath | Out-Null }

$logInstall = Join-Path $RegPath "INSTALL_LOG.txt"
$lum = Join-Path $RegPath "LAST_UPDATE.txt"
$initLine = "[INIT] $NowStamp | Host=$HostName | Dossiers=$createdDirs | Fichiers=$createdFiles | Tools=3 (verify/fix/launcher)"

# Evite les doublons d'entrees [INIT] dans le log
if (Test-Path $logInstall) {
    $logLines = Get-Content $logInstall -Encoding UTF8
    $logLines = $logLines | Where-Object { $_ -notmatch "^\[INIT\]" }
    Write-Utf8NoBom -Path $logInstall -Content ($logLines -join "`r`n")
}

Add-Content -Path $logInstall -Value $initLine
Write-Utf8NoBom -Path $lum -Content "Derniere mise a jour : $NowStamp"

$lum = Join-Path $RegPath "LAST_UPDATE.txt"
$initLine = "[INIT] $NowStamp | Host=$HostName | Dossiers=$createdDirs | Fichiers=$createdFiles"
Add-Content -Path $logInstall -Value $initLine
Write-Utf8NoBom -Path $lum -Content "Derniere mise a jour : $NowStamp"

# --- Verification automatique post-install ---
Write-Host "`n[INFO] Verification de la structure..." -ForegroundColor Cyan
$verifyReport = Join-Path $RegPath "VERIFY_AFTER_INSTALL.txt"
$dirsExist = $true
$filesExist = $true

foreach ($d in $dirs) {
    if (-not (Test-Path $d)) { $dirsExist = $false }
}
foreach ($f in $docFiles.Keys) {
    $path = Join-Path $DocPath $f
    if (-not (Test-Path $path)) { $filesExist = $false }
}

$report = @()
if ($dirsExist) { $report += "[CHECK] Dossiers : OK" } else { $report += "[WARN] Dossiers manquants" }
if ($filesExist) { $report += "[CHECK] Fichiers : OK" } else { $report += "[WARN] Fichiers manquants" }

$toolsPath = "$DocPath\scripts_utilitaires"
if ((Test-Path (Join-Path $toolsPath "verify_encoding_gouvernance_fixed.ps1")) -and
    (Test-Path (Join-Path $toolsPath "fix_encoding_gouvernance_fixed.ps1"))) {
    $report += "[CHECK] Scripts utilitaires : OK"
} else {
    $report += "[INFO] Scripts utilitaires : absents (a copier manuellement)"
}

if (Test-Path (Join-Path $toolsPath "run_encoding_maintenance.ps1")) {
    $report += "[CHECK] Launcher : OK"
} else {
    $report += "[INFO] Launcher : absent"
}


$report += "[RESULT] Verification terminee le $NowStamp"
Write-Utf8NoBom -Path $verifyReport -Content ($report -join "`r`n")

foreach ($line in $report) { Write-Host $line }

Write-Host "`n[OK] Setup complet de ChatGPT-Gouvernance-Projets termine." -ForegroundColor Green
Write-Host "Rapport : $verifyReport"
