#!/bin/sh
# ============================================================
# Script : autobrr_push_sonarr.sh
# Version : 2.2.1 (log dir auto-detect + sélection adaptative)
# Projet : Seedbox Manager – Autobrr → Sonarr
# ============================================================
# Entrées :
#   $1 = {{ .TorrentPath }}
#   $2 = {{ .TorrentUrl }}
#   $3 = {{ .ReleaseTitle }}   (facultatif)
# Flux :
#   - Identifie le .torrent archivé par Autobrr dans /autobrr_web ou /data/autobrr_web
#   - Construit l’URL publique vers le mini-serveur web (port 8990)
#   - Pousse la release à Sonarr via /api/v3/release/push
# ============================================================

SONARR_URL="http://192.168.0.101:8989"
SONARR_API_KEY="317e978a981b461f808f8cd4a370ea94"

CANDIDATE_DIRS="
/autobrr_web/torrents/sonarr
/data/autobrr_web/torrents/sonarr
"

WEB_BASE_URL="http://192.168.0.101:8990/torrents/sonarr"

# --- Log dir auto-detect: prefer /scripts/autobrr/log (singular), fallback to /scripts/autobrr/logs ---
if [ -d /scripts/autobrr/log ]; then
  LOG_DIR="/scripts/autobrr/log"
elif [ -d /scripts/autobrr/logs ]; then
  LOG_DIR="/scripts/autobrr/logs"
else
  LOG_DIR="/scripts/autobrr/log"
  mkdir -p "$LOG_DIR"
fi
LOG_FILE="$LOG_DIR/autobrr_push_sonarr.log"
MAX_LOG_BYTES=$((10 * 1024 * 1024))

TORRENT_PATH_IN="$1"
TORRENT_URL_IN="$2"
RELEASE_TITLE="$3"

# --- Logging helpers ---
mkdir -p "$(dirname "$LOG_FILE")"
if [ -f "$LOG_FILE" ]; then
  sz="$(wc -c < "$LOG_FILE" 2>/dev/null || echo 0)"
  [ "$sz" -gt "$MAX_LOG_BYTES" ] && mv "$LOG_FILE" "${LOG_FILE}.1"
fi
log() { printf "%s %s\n" "$(date '+[%Y-%m-%d %H:%M:%S]')" "$*" | tee -a "$LOG_FILE" ; }

# --- Recherche fichiers ---
find_recent_torrents() {
  for d in $CANDIDATE_DIRS; do
    [ -d "$d" ] || continue
    find "$d" -type f -name "autobrr-*.torrent" -mmin -5 2>/dev/null
  done
}

find_older_torrents() {
  for d in $CANDIDATE_DIRS; do
    [ -d "$d" ] || continue
    find "$d" -type f -name "autobrr-*.torrent" -mtime -1 2>/dev/null
  done
}

select_best_torrent() {
  # 1. Chemin direct si fourni
  if [ -n "$TORRENT_PATH_IN" ] && [ -f "$TORRENT_PATH_IN" ]; then
    echo "$TORRENT_PATH_IN"; return 0
  fi
  # 2. Fenêtre 5 min
  candidates="$(find_recent_torrents | sort -r)"
  [ -z "$candidates" ] && candidates="$(find_older_torrents | sort -r | head -n 10)"
  # 3. Match par titre si dispo
  if [ -n "$RELEASE_TITLE" ]; then
    prefix="$(echo "$RELEASE_TITLE" | cut -c1-20)"
    match="$(echo "$candidates" | grep -i "$prefix" | head -n 1)"
    [ -n "$match" ] && echo "$match" && return 0
  fi
  # 4. Sinon, le plus récent
  echo "$candidates" | head -n 1
}

log "[INFO] Début du traitement Sonarr"
[ -n "$TORRENT_URL_IN" ] && log "[INFO] URL d'origine : $TORRENT_URL_IN"
[ -n "$RELEASE_TITLE" ] && log "[INFO] Release : $RELEASE_TITLE"

TORRENT_FILE="$(select_best_torrent)"
if [ -z "$TORRENT_FILE" ] || [ ! -f "$TORRENT_FILE" ]; then
  log "[ERROR] Aucun fichier .torrent trouvé (5 min + fallback 24h)"
  exit 1
fi

BASENAME="$(basename "$TORRENT_FILE")"
HUMAN="$(ls -lh "$TORRENT_FILE" 2>/dev/null | awk '{print $5}')"
PUBLIC_URL="$WEB_BASE_URL/$BASENAME"

log "[INFO] Torrent sélectionné : $BASENAME ($HUMAN)"
log "[INFO] URL publique exposée : $PUBLIC_URL"

# --- Push Sonarr ---
PAYLOAD=$(printf '{"title":"AutobrrPush","downloadUrl":"%s","protocol":"torrent"}' "$PUBLIC_URL")
HTTP_CODE="$(curl -s -o /dev/null -w "%{http_code}" \
  -X POST "$SONARR_URL/api/v3/release/push" \
  -H "X-Api-Key: $SONARR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "$PAYLOAD")"

if [ "$HTTP_CODE" = "200" ]; then
  log "[OK] Push Sonarr réussi."
else
  log "[ERROR] Sonarr a renvoyé HTTP $HTTP_CODE"
fi

echo "[OK] Script autobrr_push_sonarr.sh installé et permissions modifiées."
exit 0
