#!/bin/bash
set -euo pipefail

LOG_DIR="/scripts/autobrr/log"
ARCHIVE_FILE="$LOG_DIR/daily_summary.log"
OUT_FILE="$LOG_DIR/weekly_summary_$(date +%G-%V).txt"

# (Optionnel) Quand tu auras Notifiarr : mets ici un webhook/URL API
NOTIFIARR_WEBHOOK="${NOTIFIARR_WEBHOOK:-}"

TODAY=$(date +%F)
CUTOFF=$(date -d "-6 days" +%F) # 7 jours glissants (aujourd’hui inclus)

mkdir -p "$LOG_DIR"

if [[ ! -f "$ARCHIVE_FILE" ]]; then
  echo "[$(date '+%F %T')] Pas d'archive quotidienne trouvée ($ARCHIVE_FILE)."
  exit 0
fi

# Filtre la fenêtre [CUTOFF..TODAY]
mapfile -t LINES < <(awk -v c="$CUTOFF" -v t="$TODAY" -F'|' '
{
  gsub(/^ +| +$/,"",$1);
  split($1, d, " ");
  if (d[1] >= c && d[1] <= t) print $0;
}' "$ARCHIVE_FILE")

# Init compteurs
declare -A day_sonarr day_radarr day_total
total_sonarr=0
total_radarr=0
total_all=0

# Préremplit les 7 jours même si vides
for offset in {6..0}; do
  d=$(date -d "-$offset days" +%F)
  day_sonarr["$d"]=0
  day_radarr["$d"]=0
  day_total["$d"]=0
done

# Parse lignes: "YYYY-MM-DD | Sonarr: X | Radarr: Y | Total: Z"
for line in "${LINES[@]}"; do
  d=$(echo "$line" | awk -F'|' '{gsub(/^ +| +$/,"",$1); print $1}')
  s=$(echo "$line" | awk -F'Sonarr: ' '{print $2}' | awk -F'|' '{gsub(/^ +| +$/,"",$1); print $1}')
  r=$(echo "$line" | awk -F'Radarr: ' '{print $2}' | awk -F'|' '{gsub(/^ +| +$/,"",$1); print $1}')
  z=$(echo "$line" | awk -F'Total: '  '{print $2}' | awk '{print $1}')
  [[ -z "$d" || -z "$s" || -z "$r" || -z "$z" ]] && continue
  day_sonarr["$d"]=$(( ${day_sonarr["$d"]} + s ))
  day_radarr["$d"]=$(( ${day_radarr["$d"]} + r ))
  day_total["$d"]=$(( ${day_total["$d"]} + z ))
  total_sonarr=$(( total_sonarr + s ))
  total_radarr=$(( total_radarr + r ))
  total_all=$(( total_all + z ))
done

# Génère le rapport texte
{
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
  echo "📊  Rapport hebdomadaire Autobrr → Sonarr/Radarr"
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
  echo "Semaine ISO : $(date +%G-%V)"
  echo "Période     : $CUTOFF → $TODAY"
  echo
  printf "   📺 Sonarr total : %3d\n" "$total_sonarr"
  printf "   🎬 Radarr total : %3d\n" "$total_radarr"
  printf "   🧹 Total global : %3d\n" "$total_all"
  echo
  echo "Détail par jour :"
  echo "Date       | Sonarr | Radarr | Total"
  echo "-----------+--------+--------+------"
  for offset in {6..0}; do
    d=$(date -d "-$offset days" +%F)
    printf "%s | %6d | %6d | %5d\n" "$d" "${day_sonarr["$d"]}" "${day_radarr["$d"]}" "${day_total["$d"]}"
  done
  echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
} | tee "$OUT_FILE"

# (Optionnel) Envoi Notifiarr plus tard : décommente et adapte
# if [[ -n "$NOTIFIARR_WEBHOOK" ]]; then
#   curl -s -X POST "$NOTIFIARR_WEBHOOK" \
#     -H "Content-Type: application/json" \
#     -d "$(jq -n \
#           --arg week "$(date +%G-%V)" \
#           --arg from "$CUTOFF" --arg to "$TODAY" \
#           --argjson s "$total_sonarr" \
#           --argjson r "$total_radarr" \
#           --argjson t "$total_all" \
#           '{title: ("Rapport hebdo Autobrr (" + $week + ")"), from:$from, to:$to, sonarr:$s, radarr:$r, total:$t }')" >/dev/null || true
#   echo "[INFO] Rapport hebdo envoyé à Notifiarr."
# fi
