#!/bin/bash
set -euo pipefail

LOG="/scripts/log/daily_qbittorrent_update.log"
MAP="/scripts/mapping_entries.txt"
STATE_DIR="/scripts/state"
ENV="/etc/systemd/system/daily_qbittorrent_update.env"
COOKIE="/tmp/qb_cookie_$$.jar"

mkdir -p "$(dirname "$LOG")" "$STATE_DIR"
touch "$LOG" "$MAP"

ts(){ date '+%F %T'; }
log(){ echo "$(ts) $*" | tee -a "$LOG"; }

# Options
DRY=false; FORCE=false; CLEAN=false; MINAGE=$((7*24*3600))
while [[ $# -gt 0 ]]; do
  case "$1" in
    --dry-run) DRY=true ;;
    --force-move) FORCE=true ;;
    --clean-orphans) CLEAN=true ;;
    --min-age) shift; MINAGE="$1" ;;
  esac; shift || true
done

# Credentials
if [[ -f "$ENV" ]]; then source "$ENV"; else log "[Error] missing $ENV"; exit 1; fi
QBT_URL="${QBITTORRENT_URL:-http://192.168.0.101:8090}"
QBT_USER="${QBITTORRENT_USER:-mousse}"
QBT_PASS="${QBITTORRENT_PASS:-password}"

# Login
resp=$(curl -s -i -c "$COOKIE" -d "username=$QBT_USER&password=$QBT_PASS" "$QBT_URL/api/v2/auth/login")
echo "$resp" | grep -q "Ok." || { log "[Error] qBittorrent auth failed"; exit 1; }
log "[Info] Connexion qBittorrent OK (dry=$DRY force=$FORCE min-age=${MINAGE}s)"

# Wrappers
qbt_info(){ curl -s -b "$COOKIE" "$QBT_URL/api/v2/torrents/info"; }
qbt_files(){ curl -s -b "$COOKIE" "$QBT_URL/api/v2/torrents/files?hash=$1"; }
qbt_setloc(){ $DRY || curl -s -b "$COOKIE" -d "hashes=$1&location=$2" "$QBT_URL/api/v2/torrents/setLocation" >/dev/null; }
qbt_recheck(){ $DRY || curl -s -b "$COOKIE" -d "hashes=$1" "$QBT_URL/api/v2/torrents/recheck" >/dev/null; }
qbt_delete(){ $DRY || curl -s -b "$COOKIE" "$QBT_URL/api/v2/torrents/delete?hashes=$1&deleteFiles=true" >/dev/null; }
progress(){ curl -s -b "$COOKIE" "$QBT_URL/api/v2/torrents/info?hashes=$1" | jq -r '.[0].progress // 0'; }

find_hash_by_old(){
  local OLD="$1"
  local h; h=$(qbt_info | jq -r --arg p "$OLD" '.[] | select(.content_path==$p) | .hash' | head -n1)
  [[ -n "$h" ]] && { echo "$h"; return 0; }
  local bn; bn=$(basename "$OLD")
  for h in $(qbt_info | jq -r '.[].hash'); do
    if qbt_files "$h" | jq -r '.[].name' | grep -Fqx "$bn"; then echo "$h"; return 0; fi
  done
  return 1
}

ensure_hardlink(){
  local SRC="$1" DST="$2"
  mkdir -p "$(dirname "$DST")"
  if [[ ! -e "$DST" ]]; then
    if $DRY; then
      log "[Dry] hardlink '$DST' → $SRC"
    else
      if ln "$SRC" "$DST" 2>/dev/null; then
        log "[Info] Hardlink créé: $DST"
      else
        log "[Error] Échec hardlink: $SRC → $DST"
      fi
    fi
  fi
}

copy_extras(){
  local HASH="$1" OLD="$2" MIRROR_ROOT="$3"
  local files dst src count=0
  files=$(qbt_files "$HASH" | jq -r '.[].name')
  [[ -z "$files" ]] && return 0
  local old_root="/data/torrents/${MIRROR_ROOT#/syno/torrents/}"
  while IFS= read -r rel; do
    [[ -z "$rel" ]] && continue
    dst="$MIRROR_ROOT/$rel"; src="$old_root/$rel"
    if [[ ! -e "$dst" && -f "$src" ]]; then
      mkdir -p "$(dirname "$dst")"
      if [[ "$rel" =~ \.(nfo|srt|sub|ass|txt|jpg|jpeg|png|sfv)$ ]]; then
        if $DRY; then log "[Dry] copy $src -> $dst"; else cp -p "$src" "$dst"; fi
        count=$((count+1))
      fi
    fi
  done <<< "$files"
  [[ $count -gt 0 ]] && log "[Info] Fichiers secondaires copiés: $count"
}

purge_mapping(){
  local T="$1" O="$2" N="$3"
  sed -i "\|^$T|$O|$N\$|d" "$MAP" || true
}

now=$(date +%s)
STATES=($(ls "$STATE_DIR"/*.state 2>/dev/null || true))
log "[Info] States trouvés: ${#STATES[@]}"

for S in "${STATES[@]}"; do
  [[ -f "$S" ]] || continue
  cat "$S" | tr -d '\r"' > "$S.tmp" && mv "$S.tmp" "$S"
  CAT=$(grep '^category=' "$S" | cut -d= -f2)
  ADDED=$(grep '^added=' "$S" | cut -d= -f2)
  UPG=$(grep '^upgrade=' "$S" | cut -d= -f2)
  OLD=$(grep '^old_path=' "$S" | cut -d= -f2-)
  NEW=$(grep '^new_path=' "$S" | cut -d= -f2-)
  [[ -z "$ADDED" || -z "$OLD" || -z "$NEW" ]] && { log "[Warn] $(basename "$S"): champs manquants"; continue; }
  AGE=$(( now - ADDED ))
  log "[Info] Processing $(basename "$S") type=$CAT age=${AGE}s upgrade=$UPG"

  if ! $FORCE && [[ $AGE -lt $MINAGE ]]; then
    log "[Skip] Trop récent (age=${AGE}s < ${MINAGE}s)"
    continue
  fi

  HASH=$(find_hash_by_old "$OLD" || true)
  [[ -z "$HASH" ]] && { log "[Warn] Torrent introuvable pour $OLD"; continue; }

  # Upgrade -> suppression
  if [[ "$UPG" -eq 1 ]]; then
    log "[Action] Upgrade → suppression torrent $HASH"
    $DRY || qbt_delete "$HASH"
    $DRY || rm -f "$S"
    purge_mapping "$CAT" "$OLD" "$NEW"
    continue
  fi

  # Chemin miroir Syno
  if [[ "$OLD" == /data/torrents/* ]]; then
    REL="${OLD#/data/torrents/}"
    MIRROR="/syno/torrents/$REL"
    ensure_hardlink "$NEW" "$MIRROR"
    copy_extras "$HASH" "$OLD" "$(dirname "$MIRROR")"
  else
    log "[Warn] OLD en dehors de /data/torrents → skip"
    continue
  fi

  # Migration qBittorrent
  ROOT=$(dirname "$MIRROR")
  log "[Action] Migration torrent hash=$HASH → $ROOT"
  qbt_setloc "$HASH" "$ROOT"
  qbt_recheck "$HASH"

  ok=0
  for i in {1..12}; do
    sleep 10
    p=$(progress "$HASH")
    log "[Info] Recheck progress=$p"
    awk "BEGIN{exit !($p >= 0.999)}" && { ok=1; break; }
  done

  if [[ $ok -eq 1 ]]; then
    log "[OK] Recheck OK → suppression $S"
    $DRY || rm -f "$S"
    purge_mapping "$CAT" "$OLD" "$NEW"
  else
    log "[Warn] Recheck échoué, state conservé"
  fi
done

rm -f "$COOKIE"
